import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { Calendar, User, Clock, ArrowRight, Search, Tag } from 'lucide-react';

const Blog = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [posts, setPosts] = useState([]);
  React.useEffect(() => {
    fetch('/public_content/data/posts.json')
      .then(r=>r.json())
      .then(d=> setPosts(d.posts||[]))
      .catch(()=> setPosts([]));
  }, []);
  const [selectedTag, setSelectedTag] = useState('all');

  const blogPosts = posts.length ? posts : [
    {
      id: 1,
      title: 'Guía Completa para el Cálculo de Pérdidas de Carga en Tuberías',
      excerpt: 'Aprende los fundamentos teóricos y prácticos para calcular correctamente las pérdidas de carga en sistemas de tuberías, incluyendo factores de fricción y accesorios.',
      content: 'Contenido completo del artículo...',
      author: 'Ing. Carlos Martínez',
      date: '2025-01-15',
      readTime: '8 min',
      tags: ['cálculos', 'hidráulica', 'tutorial'],
      image: 'https://via.placeholder.com/600x300/3B82F6/FFFFFF?text=Perdidas+de+Carga'
    },
    {
      id: 2,
      title: 'Nuevas Funcionalidades en TUBEROYIYO v2.1',
      excerpt: 'Descubre las últimas mejoras y nuevas herramientas añadidas en la versión 2.1 de TUBEROYIYO, incluyendo el módulo de análisis de redes complejas.',
      content: 'Contenido completo del artículo...',
      author: 'Equipo TUBEROYIYO',
      date: '2025-01-12',
      readTime: '5 min',
      tags: ['actualización', 'características', 'software'],
      image: 'https://via.placeholder.com/600x300/1E40AF/FFFFFF?text=Version+2.1'
    },
    {
      id: 3,
      title: 'Materiales de Tubería: Comparativa Técnica 2025',
      excerpt: 'Análisis detallado de los diferentes materiales utilizados en instalaciones de fontanería y calefacción, con sus ventajas, desventajas y aplicaciones recomendadas.',
      content: 'Contenido completo del artículo...',
      author: 'Dra. Ana Rodríguez',
      date: '2025-01-08',
      readTime: '12 min',
      tags: ['materiales', 'comparativa', 'técnico'],
      image: 'https://via.placeholder.com/600x300/2563EB/FFFFFF?text=Materiales+2025'
    },
    {
      id: 4,
      title: 'Optimización de Redes de Distribución de Agua',
      excerpt: 'Estrategias y metodologías para optimizar el diseño de redes de distribución de agua, minimizando costos y maximizando la eficiencia hidráulica.',
      content: 'Contenido completo del artículo...',
      author: 'Ing. Miguel Santos',
      date: '2025-01-05',
      readTime: '10 min',
      tags: ['optimización', 'redes', 'distribución'],
      image: 'https://via.placeholder.com/600x300/1D4ED8/FFFFFF?text=Optimizacion+Redes'
    },
    {
      id: 5,
      title: 'Casos de Estudio: Instalaciones Industriales Complejas',
      excerpt: 'Revisión de casos reales de instalaciones industriales complejas, analizando los desafíos encontrados y las soluciones implementadas.',
      content: 'Contenido completo del artículo...',
      author: 'Ing. Laura Fernández',
      date: '2025-01-02',
      readTime: '15 min',
      tags: ['casos-estudio', 'industrial', 'análisis'],
      image: 'https://via.placeholder.com/600x300/1E3A8A/FFFFFF?text=Casos+Estudio'
    },
    {
      id: 6,
      title: 'Normativas y Códigos Técnicos Actualizados',
      excerpt: 'Resumen de las últimas actualizaciones en normativas técnicas para instalaciones de fontanería y calefacción en España y Europa.',
      content: 'Contenido completo del artículo...',
      author: 'Equipo Técnico',
      date: '2024-12-28',
      readTime: '7 min',
      tags: ['normativas', 'códigos', 'actualización'],
      image: 'https://via.placeholder.com/600x300/3730A3/FFFFFF?text=Normativas+2025'
    }
  ];

  const allTags = ['all', ...new Set(blogPosts.flatMap(post => post.tags))];

  const filteredPosts = blogPosts.filter(post => {
    const matchesSearch = post.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         post.excerpt.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesTag = selectedTag === 'all' || post.tags.includes(selectedTag);
    return matchesSearch && matchesTag;
  });

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <motion.div 
          className="text-center mb-12"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
        >
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Blog Técnico
          </h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Artículos, tutoriales y noticias sobre ingeniería de tuberías, cálculos hidráulicos y las últimas novedades de TUBEROYIYO
          </p>
        </motion.div>

        {/* Search and Filters */}
        <motion.div 
          className="bg-white rounded-2xl p-6 mb-8 shadow-sm"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="flex flex-col md:flex-row gap-4">
            {/* Search */}
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Buscar artículos..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-10 pr-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/30 focus:border-blue-500"
              />
            </div>

            {/* Tags Filter */}
            <div className="flex flex-wrap gap-2">
              {allTags.map((tag) => (
                <button
                  key={tag}
                  onClick={() => setSelectedTag(tag)}
                  className={`px-4 py-2 rounded-xl font-medium text-sm transition-all duration-300 ${
                    selectedTag === tag
                      ? 'bg-blue-600 text-white shadow-md'
                      : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                  }`}
                >
                  {tag === 'all' ? 'Todos' : tag}
                </button>
              ))}
            </div>
          </div>
        </motion.div>

        {/* Blog Posts Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {filteredPosts.map((post, index) => (
            <motion.article
              key={post.id}
              className="bg-white rounded-2xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300 border border-gray-200"
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: index * 0.1 }}
              whileHover={{ y: -5 }}
            >
              <div className="relative overflow-hidden">
                <img
                  src={post.image}
                  alt={post.title}
                  className="w-full h-48 object-cover transition-transform duration-300 hover:scale-105"
                />
                <div className="absolute top-4 left-4">
                  <div className="flex flex-wrap gap-1">
                    {post.tags.slice(0, 2).map((tag) => (
                      <span
                        key={tag}
                        className="px-2 py-1 bg-blue-600/90 text-white text-xs rounded-lg font-medium"
                      >
                        {tag}
                      </span>
                    ))}
                  </div>
                </div>
              </div>

              <div className="p-6">
                <div className="flex items-center gap-4 text-sm text-gray-500 mb-3">
                  <div className="flex items-center gap-1">
                    <Calendar className="w-4 h-4" />
                    {new Date(post.date).toLocaleDateString('es-ES')}
                  </div>
                  <div className="flex items-center gap-1">
                    <Clock className="w-4 h-4" />
                    {post.readTime}
                  </div>
                </div>

                <h2 className="text-xl font-bold text-gray-900 mb-3 line-clamp-2">
                  {post.title}
                </h2>

                <p className="text-gray-600 mb-4 line-clamp-3">
                  {post.excerpt}
                </p>

                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <User className="w-4 h-4 text-gray-400" />
                    <span className="text-sm text-gray-600">{post.author}</span>
                  </div>

                  <motion.button
                    className="inline-flex items-center gap-2 text-blue-600 hover:text-blue-700 font-medium text-sm"
                    whileHover={{ x: 5 }}
                  >
                    Leer más
                    <ArrowRight className="w-4 h-4" />
                  </motion.button>
                </div>
              </div>
            </motion.article>
          ))}
        </div>

        {filteredPosts.length === 0 && (
          <motion.div 
            className="text-center py-12"
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.5 }}
          >
            <Search className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-gray-600 mb-2">
              No se encontraron artículos
            </h3>
            <p className="text-gray-500">
              Intenta con otros términos de búsqueda o selecciona una etiqueta diferente
            </p>
          </motion.div>
        )}

        {/* AdSense Placeholder */}
        <div className="mt-12">
          <div className="bg-gray-100 border-2 border-dashed border-gray-300 rounded-2xl p-8 text-center">
            <span className="text-gray-500">Espacio publicitario - Google AdSense</span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Blog;