import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { Download, FileText, Image, File, Calendar, Eye } from 'lucide-react';

const Downloads = () => {
  const [selectedCategory, setSelectedCategory] = useState('all');
  const [files, setFiles] = useState([]);
  React.useEffect(()=>{
    fetch('/public_content/data/files.json')
      .then(r=>r.json())
      .then(d=> setFiles(d.files||[]))
      .catch(()=> setFiles([]));
  },[]);

  const downloadFiles = files.length ? files.map(f=>({id:f.id,name:f.name,description:'Archivo subido',type:(f.mime||'file').split('/')[1]||'file',category:'general',size:(Math.round((f.size||0)/1024)+' KB'),date:f.date||'',link:f.stored})) : [
    {
      id: 1,
      name: 'Manual de Usuario TUBEROYIYO v2.1',
      description: 'Guía completa para el uso de todas las funcionalidades de la aplicación',
      type: 'pdf',
      category: 'manual',
      size: '2.5 MB',
      date: '2025-01-15',
      downloads: 1250
    },
    {
      id: 2,
      name: 'Tabla de Pérdidas de Carga',
      description: 'Tabla técnica con valores de pérdidas de carga para diferentes materiales',
      type: 'pdf',
      category: 'tabla',
      size: '850 KB',
      date: '2025-01-10',
      downloads: 890
    },
    {
      id: 3,
      name: 'Plano Ejemplo - Instalación Residencial',
      description: 'Ejemplo práctico de instalación de fontanería en vivienda unifamiliar',
      type: 'dxf',
      category: 'plano',
      size: '1.2 MB',
      date: '2025-01-08',
      downloads: 567
    },
    {
      id: 4,
      name: 'Catálogo de Materiales 2025',
      description: 'Catálogo actualizado con especificaciones técnicas de tuberías y accesorios',
      type: 'pdf',
      category: 'catalogo',
      size: '4.1 MB',
      date: '2025-01-05',
      downloads: 723
    },
    {
      id: 5,
      name: 'Plantilla de Cálculo Excel',
      description: 'Hoja de cálculo complementaria para verificación de resultados',
      type: 'xlsx',
      category: 'plantilla',
      size: '650 KB',
      date: '2025-01-03',
      downloads: 445
    },
    {
      id: 6,
      name: 'Imágenes de Referencia - Conexiones',
      description: 'Galería de imágenes con tipos de conexiones y accesorios',
      type: 'zip',
      category: 'imagen',
      size: '3.2 MB',
      date: '2024-12-28',
      downloads: 334
    }
  ];

  const categories = [
    { key: 'all', label: 'Todos los archivos', count: downloadFiles.length },
    { key: 'manual', label: 'Manuales', count: downloadFiles.filter(f => f.category === 'manual').length },
    { key: 'tabla', label: 'Tablas Técnicas', count: downloadFiles.filter(f => f.category === 'tabla').length },
    { key: 'plano', label: 'Planos DXF', count: downloadFiles.filter(f => f.category === 'plano').length },
    { key: 'catalogo', label: 'Catálogos', count: downloadFiles.filter(f => f.category === 'catalogo').length },
    { key: 'plantilla', label: 'Plantillas', count: downloadFiles.filter(f => f.category === 'plantilla').length },
    { key: 'imagen', label: 'Imágenes', count: downloadFiles.filter(f => f.category === 'imagen').length }
  ];

  const getFileIcon = (type) => {
    switch (type) {
      case 'pdf': return FileText;
      case 'dxf': return File;
      case 'xlsx': return FileText;
      case 'zip': return Image;
      default: return File;
    }
  };

  const getFileColor = (type) => {
    switch (type) {
      case 'pdf': return 'from-red-500 to-red-600';
      case 'dxf': return 'from-blue-500 to-blue-600';
      case 'xlsx': return 'from-green-500 to-green-600';
      case 'zip': return 'from-purple-500 to-purple-600';
      default: return 'from-gray-500 to-gray-600';
    }
  };

  const filteredFiles = selectedCategory === 'all' 
    ? downloadFiles 
    : downloadFiles.filter(file => file.category === selectedCategory);

  const handleDownload = (file) => {
    // Simular descarga
    console.log(`Descargando: ${file.name}`);
    // Aquí se implementaría la lógica real de descarga
  };

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <motion.div 
          className="text-center mb-12"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
        >
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Documentos y Descargas
          </h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Accede a manuales, tablas técnicas, planos de ejemplo y recursos complementarios para TUBEROYIYO
          </p>
        </motion.div>

        {/* Category Filters */}
        <motion.div 
          className="bg-white rounded-2xl p-6 mb-8 shadow-sm"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="flex flex-wrap gap-2">
            {categories.map((category, index) => (
              <motion.button
                key={category.key}
                onClick={() => setSelectedCategory(category.key)}
                className={`px-4 py-2 rounded-xl font-medium text-sm transition-all duration-300 ${
                  selectedCategory === category.key
                    ? 'bg-blue-600 text-white shadow-md'
                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                }`}
                whileHover={{ scale: 1.05 }}
                whileTap={{ scale: 0.95 }}
                initial={{ opacity: 0, x: 20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.05 }}
              >
                {category.label}
                <span className="ml-2 px-2 py-0.5 bg-black/10 rounded-lg text-xs">
                  {category.count}
                </span>
              </motion.button>
            ))}
          </div>
        </motion.div>

        {/* External Integration Container */}
        <div id="app-descargas-container" className="mb-8">
          {/* Este contenedor está disponible para futuras integraciones externas */}
        </div>

        {/* Files Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredFiles.map((file, index) => {
            const FileIcon = getFileIcon(file.type);
            const fileColor = getFileColor(file.type);
            
            return (
              <motion.div
                key={file.id}
                className="bg-white rounded-2xl p-6 shadow-sm hover:shadow-lg transition-all duration-300 border border-gray-200"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.5, delay: index * 0.1 }}
                whileHover={{ y: -5 }}
              >
                <div className="flex items-start gap-4 mb-4">
                  <div className={`p-3 bg-gradient-to-br ${fileColor} rounded-xl`}>
                    <FileIcon className="w-6 h-6 text-white" />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h3 className="font-semibold text-gray-900 mb-1 line-clamp-2">
                      {file.name}
                    </h3>
                    <p className="text-sm text-gray-600 mb-2 line-clamp-2">
                      {file.description}
                    </p>
                  </div>
                </div>

                <div className="flex items-center gap-4 text-sm text-gray-500 mb-4">
                  <div className="flex items-center gap-1">
                    <Calendar className="w-4 h-4" />
                    {new Date(file.date).toLocaleDateString('es-ES')}
                  </div>
                  <div className="flex items-center gap-1">
                    <Eye className="w-4 h-4" />
                    {file.downloads}
                  </div>
                </div>

                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-gray-600">
                    {file.size} • {file.type.toUpperCase()}
                  </span>
                  <motion.button
                    onClick={() => handleDownload(file)}
                    className="inline-flex items-center gap-2 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium text-sm transition-colors"
                    whileHover={{ scale: 1.05 }}
                    whileTap={{ scale: 0.95 }}
                  >
                    <Download className="w-4 h-4" />
                    Descargar
                  </motion.button>
                </div>
              </motion.div>
            );
          })}
        </div>

        {filteredFiles.length === 0 && (
          <motion.div 
            className="text-center py-12"
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.5 }}
          >
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-gray-600 mb-2">
              No hay archivos en esta categoría
            </h3>
            <p className="text-gray-500">
              Selecciona otra categoría para ver más archivos disponibles
            </p>
          </motion.div>
        )}

        {/* AdSense Placeholder */}
        <div className="mt-12">
          <div className="bg-gray-100 border-2 border-dashed border-gray-300 rounded-2xl p-8 text-center">
            <span className="text-gray-500">Espacio publicitario - Google AdSense</span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Downloads;